/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2006 Sandia National Laboratories.  Developed at the
    University of Wisconsin--Madison under SNL contract number
    624796.  The U.S. Government and the University of Wisconsin
    retian certain rights to this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    (2006) kraftche@cae.wisc.edu
   
  ***************************************************************** */


#include "Mesquite_all_headers.hpp"
#include <iostream>
#include <sstream>
#include "MeshBoundaryDomain2D.cpp"

#include "CompositeOFExponent.hpp"
#include "CompositeOFNLog.hpp"

using std::cout;
using std::cerr;
using std::endl;
using std::ostream;
using std::ostringstream;

using namespace MESQUITE_NS;


int do_smoother( const char* input_file, 
                 ostringstream &outputStream,
		 Mesquite::TMetric* targetMetric,
		 int pOrder)
{

    int mNumInterfaceSmoothIters =10;
    bool mFixBndryInInterfaceSmooth = false; //this fixes exterior surface nodes
    bool project_gradient= false;
    double cos_crease_angle=0.2;
    MsqPrintError err(cerr);
   
    //
    // Mesh we will be optimizing
    //
    MeshImpl mesh, deformedMesh;;
    mesh.read_vtk( input_file, err );
    deformedMesh.read_vtk("/g/g92/kim74/mesquite-2.99/mesquite-debug/DeformingMeshes/cylinder_aniso_3_1_inverted_two_dimension_15.vtk", err);

    if( err ) cout<<err<<endl;

    static bool firstCall=true;
    if( firstCall )
    {
	mesh.write_vtk("InitialMesh.vtk",err);
	firstCall=false;
    }

    // get all vertices
    std::vector<Mesquite::Mesh::VertexHandle> vertices;
    deformedMesh.get_all_vertices(vertices, err);
    if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
    int num_vertices = vertices.size();
    
    // get application fixed vertices
    std::vector<bool> app_fixed(num_vertices);

    deformedMesh.vertices_get_fixed_flag(&(vertices[0]), app_fixed, num_vertices, err);
    if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}

    // create planar domain for interior and assessor queues
    MESQUITE_NS::XYPlanarDomain geom;

    // create boundary domain and find mesh boundary
    Mesquite::MeshBoundaryDomain2D* mesh_domain = new Mesquite::MeshBoundaryDomain2D( MeshBoundaryDomain2D::XY, 0.0, project_gradient, 
                                                                                     Mesquite::MeshBoundaryDomain2D::QUADRATIC);
    mesh_domain->skin_area_mesh(&deformedMesh,cos_crease_angle,"material");

    std::vector<MESQUITE_NS::Mesh::VertexHandle> theBoundaryVertices;
    mesh_domain->get_boundary_vertices( theBoundaryVertices );
    
    int num_boundary_vertices = theBoundaryVertices.size();

    std::vector<MESQUITE_NS::Mesh::VertexHandle> theBoundaryEdges;
    mesh_domain->get_boundary_edges( theBoundaryEdges );

    //     int num_boundary_edges = theBoundaryEdges.size();
    std::vector<bool> fixed_flags_boundary(num_boundary_vertices);

    // get application fixed boundary vertices
    std::vector<bool> app_fixed_boundary(num_boundary_vertices);
    deformedMesh.vertices_get_fixed_flag(&(theBoundaryVertices[0]),app_fixed_boundary,num_boundary_vertices, err);

    // only fix  boundary vertices along corners
    int num_fixed_boundary_flags = 0;

    std::vector<MESQUITE_NS::Mesh::VertexHandle> theCornerVertices;
    mesh_domain->get_corner_vertices( theCornerVertices );

    // fix only vertices that are classified as corners
    for (int i = 0; i < num_boundary_vertices; i++)
    {
      //
      // if we allow boundary vertices to float, we do so here
      //
        if (!mFixBndryInInterfaceSmooth)
            fixed_flags_boundary[i] = false;
        else
            fixed_flags_boundary[i] = app_fixed_boundary[i];

        for (int j = 0; j < theCornerVertices.size(); j++)
        {
            if (theCornerVertices[j] == theBoundaryVertices[i])
            {
                fixed_flags_boundary[i] = true;
                num_fixed_boundary_flags++;
                break;
            }    
        }
    }
    printf("fixed %d of %d boundary vertices (those classified corner)\n", num_fixed_boundary_flags, num_boundary_vertices);


    // creates three intruction queues
    Mesquite::InstructionQueue boundary_queue;
    Mesquite::InstructionQueue interior_queue;

  IdealShapeTarget tc;
 //   ReferenceMesh refMesh( &mesh );
 //  RefMeshTargetCalculator W( &refMesh, false );
 TShapeNB1 target_metric; 
//  TShapeSizeNB3 target_metric;
//   TShapeSize2DNB1 target_metric;
//  ReferenceMesh refMesh( &mesh );
//  RefMeshTargetCalculator W( &refMesh, false );

 TQualityMetric mu( &tc, &target_metric ); 


 //   TQualityMetric metric( &tc, targetMetric );
    UntangleBetaQualityMetric nu(0.01);
  //   SizeMetric nu;
	LPtoPTemplate obj_func1(&mu, 2, err);
	LPtoPTemplate obj_func2(&nu, 2, err);


 //   Mesquite::LPtoPTemplate *obj_func1 = new Mesquite::LPtoPTemplate(&metric, pOrder, err);
 //   Mesquite::LPtoPTemplate *obj_func2 = new Mesquite::LPtoPTemplate(&nu, pOrder, err);

  CompositeOFAdd obj_func(&obj_func1, &obj_func2, 0);




    Mesquite::ConjugateGradient* boundary_alg = new Mesquite::ConjugateGradient(&obj_func);
    boundary_alg->use_element_on_vertex_patch();

    Mesquite::ConjugateGradient* interior_alg = new Mesquite::ConjugateGradient(&obj_func );
    interior_alg->use_element_on_vertex_patch();

    // **************Set stopping criterion**************
  //  double grad_norm = 1e-7;
  //  double successiveEps = 1e-7;
    int boundary_outer = 1;
    int boundary_inner = 1;
    
    // for boundary
    Mesquite::TerminationCriterion* boundaryTermInner = new Mesquite::TerminationCriterion();
    Mesquite::TerminationCriterion* boundaryTermOuter = new Mesquite::TerminationCriterion();
    // boundaryTermInner->add_absolute_gradient_L2_norm(grad_norm);
    // boundaryTermInner->add_relative_successive_improvement(successiveEps);
  //  boundaryTermOuter->add_relative_successive_improvement(successiveEps);
    boundaryTermOuter->add_iteration_limit(boundary_outer);
    boundaryTermInner->add_iteration_limit(boundary_inner);

    boundary_alg->set_outer_termination_criterion(boundaryTermOuter);
    boundary_alg->set_inner_termination_criterion(boundaryTermInner);

    // for interior
    Mesquite::TerminationCriterion* interiorTermInner = new Mesquite::TerminationCriterion();
    Mesquite::TerminationCriterion* interiorTermOuter = new Mesquite::TerminationCriterion();
 //   interiorTermInner->add_absolute_gradient_L2_norm(grad_norm);
 //   interiorTermInner->add_relative_successive_improvement(successiveEps);
    // interiorTermInner->add_iteration_limit(3); // for element_on_vertex_patch mode
    interiorTermInner->add_iteration_limit(1); // for global_patch mode

    interiorTermOuter->add_iteration_limit(40);
    interior_alg->set_outer_termination_criterion(interiorTermOuter);
    interior_alg->set_inner_termination_criterion(interiorTermInner);

    // ConditionNumberQualityMetric qm_metric;
     QualityAssessor boundary_assessor,interior_assessor;

    // boundary_assessor.add_quality_assessment( &metric, 10 );
    // boundary_assessor.add_quality_assessment( &qm_metric );

    // interior_assessor.add_quality_assessment( &metric, 10 );
     interior_assessor.add_quality_assessment( &mu );


    // set the boundary instruction queue
    // boundary_queue.add_quality_assessor( &boundary_assessor, err );
    boundary_queue.set_master_quality_improver(boundary_alg, err);
    // boundary_queue.add_quality_assessor( &boundary_assessor, err );

    interior_queue.add_quality_assessor( &interior_assessor, err );
    // set the interior instruction queue
    // interior_queue.add_quality_assessor( &interior_assessor, err );
    interior_queue.set_master_quality_improver(interior_alg, err);
    // interior_queue.add_quality_assessor( &interior_assessor, err );
    interior_queue.add_quality_assessor( &interior_assessor, err );

    err.clear();

    //
    // we repeatedly change which flags are fixed in these iterations
    //
    std::vector<bool> fixed_flags(num_vertices);

    for (int j=0; j<mNumInterfaceSmoothIters; j++) {

	cout<<" Boundary + Interior smoothing pass "<< j<<"....."<<endl;
        
        // smooth boundary only
        for (int i = 0; i < num_vertices; i++) fixed_flags[i] = true;
        deformedMesh.vertices_set_fixed_flag(&(vertices[0]),fixed_flags,num_vertices, err);

        deformedMesh.vertices_set_fixed_flag(&(theBoundaryVertices[0]),fixed_flags_boundary,num_boundary_vertices, err);

        boundary_queue.run_instructions(&deformedMesh, mesh_domain, err);
        cout<<" boundary smooth completed in "<<boundaryTermOuter->get_iteration_count()<<" outer and "<<boundaryTermInner->get_iteration_count()<<" inner iterations."<<endl;
        //
        // smooth interior only
	//
        if (!mFixBndryInInterfaceSmooth)
        {
            //
            // let all interior vertices float during the interior smooth.
            //
            for (int i = 0; i < num_vertices; i++) fixed_flags[i] = false;
            deformedMesh.vertices_set_fixed_flag(&(vertices[0]),fixed_flags,num_vertices, err);
        }
        else
        {
            //
            // use the app_fixed settings for the fixed state of boundary vertices.
            //
            deformedMesh.vertices_set_fixed_flag(&(vertices[0]),app_fixed,num_vertices, err);      
        }
        if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
        for (int i = 0; i < num_boundary_vertices; i++) fixed_flags[i] = true;

        deformedMesh.vertices_set_fixed_flag(&(theBoundaryVertices[0]),fixed_flags,num_boundary_vertices, err);
        if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}

        interior_queue.run_instructions(&deformedMesh, &geom, err);
        if (MSQ_CHKERR(err)) {std::cout << err << std::endl; exit(EXIT_FAILURE);}
        cout<<" interior smooth completed in "<<interiorTermOuter->get_iteration_count()<<" outer and "<<interiorTermInner->get_iteration_count()<<" inner iterations."<<endl;

    }

    


    outputStream<<"_"<<targetMetric->get_name()<<"_p"<<pOrder<<".vtk";
    deformedMesh.write_vtk( outputStream.str().c_str(), err );
    if (MSQ_CHKERR(err)) return 2;
    cout << "Wrote: " << outputStream.str() << endl;





    return 0;
}



int main( int argc, char* argv[] )
{
    MsqPrintError err(cout);
    
    // MsqDebug::enable(MsqDebug::WARN);
    // MsqDebug::enable(MsqDebug::INFO);

    ostringstream meshfile,infile, outfile;

    meshfile<<"/g/g92/kim74/mesquite-2.99/mesquite-debug/DeformingMeshes/cylinder_initial_refined.vtk";



    std::vector<TMetric*> vecOfMetrics;
//    vecOfMetrics.push_back( new TShapeB1 ); // converges fast, results not so great

    vecOfMetrics.push_back( new TShapeNB1 ); // significant changes, shape does improve
//    vecOfMetrics.push_back( new TShape2DNB2 ); // no diff from initial mesh ?!?
//    vecOfMetrics.push_back( new TShapeSizeNB3 );  // decent results
    // vecOfMetrics.push_back( new TShapeSizeB1 ); // odd shaped elements behind circle, maybe not converged enough?
    // vecOfMetrics.push_back( new TShapeSizeOrientB2 );  // decent except for region behind circle with very poor elements


    // vecOfMetrics.push_back( new TSizeNB1 );  // slow convergence
    // vecOfMetrics.push_back( new TSizeB1 );  // slow convergence
    // vecOfMetrics.push_back( new TShapeSizeB3 );  // doesn't converge
    // vecOfMetrics.push_back( new TShapeSizeOrientB1 );  // doesn't converge
    // vecOfMetrics.push_back( new TShape2DB1 );  // this target doesn't exist in default version
    // vecOfMetrics.push_back( new TShapeSize2DB2 );  // doesn't converge
    // vecOfMetrics.push_back( new TShapeSize2DNB1 );  // doesn't converge
    // vecOfMetrics.push_back( new TShapeSize2DNB2 );  // doesn't converge
    // vecOfMetrics.push_back( new TShapeSizeOrientNB1 );  // doesn't converge

    int result;
    for(int pOrder=2;pOrder <=2;pOrder++)
    {
      for(std::vector<Mesquite::TMetric*>::iterator it = vecOfMetrics.begin(); 
	  it != vecOfMetrics.end(); ++it)
      {
       
	ostringstream outfile;
	outfile << "SmoothedMesh";
	
	result = do_smoother( meshfile.str().c_str(), outfile, *it, pOrder);
      }
    }
            
    for(std::vector<TMetric*>::iterator it= vecOfMetrics.begin(); it !=vecOfMetrics.end(); ++it){
      delete *it;
    }
    
            
    return result;
}

